from __future__ import annotations

import subprocess
from typing import Dict, List

try:
    import psutil  # type: ignore
except ImportError:  # pragma: no cover
    psutil = None


def get_services_snapshot() -> Dict[str, List[str]]:
    snapshot: Dict[str, List[str]] = {"running": [], "stopped": [], "disabled": []}
    if not psutil or not hasattr(psutil, "win_service_iter"):
        return snapshot

    for service in psutil.win_service_iter():
        try:
            status = service.status().lower()
            start_type = service.start_type().lower()
            display_name = service.display_name()

            if status == "running":
                snapshot["running"].append(display_name)
            elif start_type == "disabled":
                snapshot["disabled"].append(display_name)
            else:
                snapshot["stopped"].append(display_name)
        except Exception:
            continue

    for key in snapshot:
        snapshot[key].sort()
    return snapshot


def configure_service_start_mode(service_name: str, start_mode: str) -> str:
    """
    Update the start mode of a service using `sc config`.
    start_mode: one of auto, demand, disabled
    """
    start_mode = start_mode.lower()
    if start_mode not in {"auto", "demand", "disabled"}:
        raise ValueError("start_mode must be 'auto', 'demand', or 'disabled'.")

    cmd = ["sc", "config", service_name, f"start={start_mode}"]
    completed = subprocess.run(cmd, capture_output=True, text=True, check=False)
    if completed.returncode != 0:
        raise RuntimeError(completed.stderr.strip() or f"Failed to configure {service_name}")
    return completed.stdout.strip()


def change_service_state(service_name: str, action: str) -> str:
    """
    Start or stop a service using `sc start` or `sc stop`.
    action: 'start' or 'stop'
    """
    action = action.lower()
    if action not in {"start", "stop"}:
        raise ValueError("action must be 'start' or 'stop'.")

    cmd = ["sc", action, service_name]
    completed = subprocess.run(cmd, capture_output=True, text=True, check=False)
    if completed.returncode != 0:
        raise RuntimeError(completed.stderr.strip() or f"Failed to {action} {service_name}")
    return completed.stdout.strip()

